/**
 * =============================================
 * 🎰 DrumRollNumber.js - ドラムロール数字表示プラグイン
 * =============================================
 *
 * このプラグインは、RPGツクールMVでスロットマシンのように
 * 数字をくるくると回して表示する演出を実現します。
 * 経験値、ゴールド、スコアなどのリザルト画面での使用に最適です。
 *
 * ✅ 表示ごとに：
 *  - 表示位置・桁数・画像・スクロール速度・効果音 を個別設定可能
 *  - 表示は順番に実行され、1つずつ確定してから次へ進む
 *  - 各桁または最終確定時にSEを鳴らす（1桁時は最終SEのみ）
 *  - 表示後はフェードアウトで消去可能
 *
 * 📌 プラグインコマンド：
 * ShowRollNumber ... 数値を右から1桁ずつくるくる表示
 * EraseRollNumber ... 指定IDの数字をフェードアウトで削除
 *
 * 📂 必要な素材：
 * 画像フォルダに「縦に0～9を並べた数字画像」を配置してください。
 * 例：32px×320px（1桁32x32）など。
 *
 * 📖 使用例：
 * ShowRollNumber 1 5 100 200 32 32 20 4 8 ScoreImage CoinStop CoinFinish
 * EraseRollNumber 1 60
 *
 * =============================================
 */
/*:
 * @target MZ
 * @plugindesc 指定した変数をドラムロールで数字表示するプラグイン（複数表示・効果音対応・フェード削除）
 * @author ChatGPT
 *
 * @command ShowRollNumber
 *
 * @arg trimLeadingZeros
 * @type boolean
 * @text 先頭の0を非表示
 * @desc 実際の数値より上位の0は非表示にします。
 * @default false
 * @text ドラムロール表示
 * @desc 指定した変数をスロット風に桁ごとに回して表示します。
 *
 * @arg id
 * @type number
 * @text 表示ID
 * @desc 表示を管理するための識別ID（複数同時制御用）
 *
 * @arg variableId
 * @type variable
 * @text 表示する変数ID
 * @desc 表示したい値が格納された変数ID
 *
 * @arg x
 * @type number
 * @text X座標
 * @desc 表示位置（右端の桁）X座標
 *
 * @arg y
 * @type number
 * @text Y座標
 * @desc 表示位置（右端の桁）Y座標
 *
 * @arg digitW
 * @type number
 * @text 桁の幅
 * @desc 1桁ごとの画像幅（例：32）
 *
 * @arg digitH
 * @type number
 * @text 桁の高さ
 * @desc 1桁ごとの画像高さ（例：32）
 *
 * @arg framePerDigit
 * @type number
 * @text 桁確定までの時間
 * @desc 各桁が止まるまでのフレーム数
 *
 * @arg digitCount
 * @type number
 * @text 表示桁数
 * @desc 表示する桁数（ゼロ埋めあり）
 *
 * @arg loopSpeed
 * @type number
 * @text 1周スピード
 * @desc 数字が0〜9を1周する速さ（フレーム）
 *
 * @arg imageName
 * @type file
 * @dir img/pictures
 * @text 数字画像名
 * @desc 縦に0〜9が並んだ画像ファイル名
 *
 * @arg seDigit
 * @type file
 * @dir audio/se
 * @text 桁確定SE
 * @desc 各桁が止まるときに鳴らす効果音（1桁時は鳴らない）
 *
 * @arg seFinal
 * @type file
 * @dir audio/se
 * @text 最終確定SE
 * @desc 最後の桁が止まったときに鳴らす効果音
 *
 * @command EraseRollNumber
 * @text 表示をフェード削除
 * @desc 指定IDの表示をフェードアウトで削除します。
 *
 * @arg id
 * @type number
 * @text 表示ID
 * @desc 削除したい表示の識別ID
 *
 * @arg duration
 * @type number
 * @text フェード時間
 * @desc フェードアウトにかける時間（フレーム）

 * @target MV
 * @plugindesc 指定した変数をドラムロールで数字表示するプラグイン（複数表示・効果音対応・フェード削除）
 * @author ChatGPT
 *
 * @help
 * ■ プラグインコマンド
 * ShowRollNumber id variableId x y digitW digitH framePerDigit digitCount loopSpeed imageName seDigit seFinal
 * EraseRollNumber id duration
 *
 * ■ 画像仕様
 * img/pictures にある画像を使用。
 * 数字は縦に0～9が並んでいる形式。1桁サイズは個別指定可能。
 */

(function() {
  // ドラムロール表示を一括管理するマネージャー
  const DrumRollManager = {
    queue: [],             // 表示待ちのキュー
    active: null,          // 現在実行中の表示
    activeMap: {},         // 表示IDと対応するRollNumberTask

    // 表示リクエストをキューに追加
    enqueue(params) {
      // 既存の同じIDの表示があれば即座に削除
      const existing = this.activeMap[params.id];
      if (existing) {
        SceneManager._scene.removeChild(existing.container);
        delete this.activeMap[params.id];
      }
      this.queue.push(params);
    },

    // 表示の更新処理、アクティブな表示とフェード中の表示を処理
    update() {
      if (!this.active && this.queue.length > 0) {
        const next = this.queue.shift();
        this.active = new RollNumberTask(next, () => {
          this.active = null;
        });
        this.activeMap[next.id] = this.active;
        SceneManager._scene.addChild(this.active.container);
      }

      if (this.active) {
        this.active.update();
      }

      for (const id in this.activeMap) {
        const task = this.activeMap[id];
        if (task._fadeMode) task.updateFade();
      }
    },

    // 表示をフェードアウトして削除する
    erase(id, duration) {
      const task = this.activeMap[id];
      if (task) {
        task.fadeOut(duration);
      }
    }
  };

  // Scene_Map の update を拡張して DrumRollManager を更新
  const _Scene_Map_update = Scene_Map.prototype.update;
  Scene_Map.prototype.update = function() {
    _Scene_Map_update.call(this);
    DrumRollManager.update();
  };

  // イベントの高速スキップを防ぐための wait モード追加
const _Game_Interpreter_updateWaitMode = Game_Interpreter.prototype.updateWaitMode;
Game_Interpreter.prototype.updateWaitMode = function() {
  if (this._drumRollBusy) return true;
  return _Game_Interpreter_updateWaitMode.call(this);
};

// ツクールMZ向けのプラグインコマンド登録
  if (PluginManager.registerCommand) {
    PluginManager.registerCommand("DrumRol NumberPlugin02", "ShowRollNumber", args => {
      const {
        id, variableId, x, y, digitW, digitH,
        framePerDigit, digitCount, loopSpeed,
        imageName, seDigit, seFinal
      } = args;

      const params = {
        id: Number(id),
        variableId: Number(variableId),
        x: Number(x),
        y: Number(y),
        digitW: Number(digitW),
        digitH: Number(digitH),
        framePerDigit: Number(framePerDigit),
        digitCount: Number(digitCount),
        loopSpeed: Number(loopSpeed),
        imageName,
        seDigit: seDigit || "",
        seFinal: seFinal || "",
        trimLeadingZeros: args.trimLeadingZeros === "true"
      };

      const required = ["id", "variableId", "x", "y", "digitW", "digitH", "framePerDigit", "digitCount", "loopSpeed"];
      for (const key of required) {
        if (isNaN(params[key])) {
          console.warn(`[DrumRoll] ShowRollNumber: ${key} が不正なためスキップします。`);
          return;
        }
      }

      const interpreter = this;
      interpreter._drumRollBusy = true;
      params.onFinish = () => {
        interpreter._drumRollBusy = false;
      };
      DrumRollManager.enqueue(params);
    });

    PluginManager.registerCommand("DrumRol NumberPlugin02", "EraseRollNumber", args => {
      const id = Number(args.id);
      const duration = Number(args.duration);
      if (isNaN(id) || isNaN(duration)) {
        console.warn("[DrumRoll] EraseRollNumber: パラメータ不正でスキップします。");
        return;
      }
      DrumRollManager.erase(id, duration);
    });
  }

// ドラムロール表示タスク（1つ分の数字表示）
  function RollNumberTask(params, onFinish) {
    this._params = params;
    this._digits = [];
    this._finished = false;
    this._onFinish = onFinish;
    this._value = $gameVariables.value(params.variableId) || 0;
    this._container = new Sprite();
    this._container.x = 0;
    this._container.y = 0;
    this._tick = 0;
    this._started = false;
    this._digitIndex = 0;
    this._digitSprites = [];
    this.prepareDigits();
  }

  // 数字を桁ごとに分割し、Sprite_RollDigit として配置
  RollNumberTask.prototype.prepareDigits = function() {
    const str = this._value.toString().padStart(this._params.digitCount, '0');
    this._digits = str.split('').map(n => Number(n));
    const baseX = this._params.x;
    const baseY = this._params.y;
    const w = this._params.digitW;
    const h = this._params.digitH;

    let hideDigits = 0;
    if (this._params.trimLeadingZeros) {
      for (let i = 0; i < this._digits.length - 1; i++) {
        if (this._digits[i] === 0) {
          hideDigits++;
        } else {
          break;
        }
      }
    }

    for (let i = 0; i < this._params.digitCount; i++) {
      const digitSprite = new Sprite_RollDigit({
        targetNumber: this._digits[this._params.digitCount - 1 - i],
        digitW: w,
        digitH: h,
        framePerDigit: this._params.framePerDigit,
        loopSpeed: this._params.loopSpeed,
        imageName: this._params.imageName,
        seName: this._params.seDigit,
        x: baseX - i * w,
        y: baseY
      }, i);
      if (this._params.trimLeadingZeros && i >= this._params.digitCount - hideDigits) {
        digitSprite.visible = false;
      }
      this._container.addChild(digitSprite);
      this._digitSprites.push(digitSprite);
    }
  };

  // 各桁の表示進行、SE再生と終了判定
  RollNumberTask.prototype.update = function() {
    if (this._finished) return;
    const current = this._digitSprites[this._digitIndex];
    if (!current) {
      this._digitIndex++;
      return;
    }
    if (!current._started) {
      current.start();
    }
    current.update();

    if (current._finished) {
      if (this._params.digitCount > 1) {
        AudioManager.playSe({ name: this._params.seDigit, volume: 90, pitch: 100, pan: 0 });
      }
      this._digitIndex++;
      if (this._digitIndex >= this._digitSprites.length) {
        AudioManager.playSe({ name: this._params.seFinal, volume: 90, pitch: 100, pan: 0 });
        this._finished = true;
        this._onFinish();
      }
    }
  };

  // フェードアウト開始
  RollNumberTask.prototype.fadeOut = function(duration) {
    if (isNaN(duration) || duration <= 0) duration = 60;
    this._fadeMode = true;
    this._fadeFrame = 0;
    this._fadeDuration = duration;
  };

  // フェード処理（徐々に透明度を下げて消去）
  RollNumberTask.prototype.updateFade = function() {
    if (!this._fadeMode) return;
    this._fadeFrame++;
    const alpha = 1.0 - this._fadeFrame / this._fadeDuration;
    this._container.opacity = Math.floor(alpha * 255);
    if (this._fadeFrame >= this._fadeDuration) {
      SceneManager._scene.removeChild(this._container);
      delete DrumRollManager.activeMap[this._params.id];
      this._fadeMode = false;
    }
  };

  // container プロパティを取得するゲッター
  Object.defineProperty(RollNumberTask.prototype, "container", {
    get: function() {
      return this._container;
    }
  });

  // 1桁の数字表示を管理するスプライトクラス
  function Sprite_RollDigit(params, index) {
    this.initialize(params, index);
  }

  Sprite_RollDigit.prototype = Object.create(Sprite.prototype);
  Sprite_RollDigit.prototype.constructor = Sprite_RollDigit;

  // 初期化処理。画像読み込みや座標設定
  Sprite_RollDigit.prototype.initialize = function(params, index) {
    Sprite.prototype.initialize.call(this);
    this._params = params;
    this._index = index;
    this._started = false;
    this._finished = false;
    this._targetNumber = params.targetNumber;
    this._frameCount = 0;
    this._bitmap = params.imageName ? ImageManager.loadPicture(params.imageName) : new Bitmap(1, 1);
    this._digitW = params.digitW;
    this._digitH = params.digitH;
    this._loopSpeed = params.loopSpeed;
    this._framePerDigit = params.framePerDigit;
    this._currentScroll = 0;
    this._loop = 0;
    this.x = params.x;
    this.y = params.y;
    this.setFrame(0, 0, this._digitW, this._digitH);
  };

  // ドラムロール開始
  Sprite_RollDigit.prototype.start = function() {
    this._started = true;
    this._frameCount = 0;
  };

  // 数字のスクロール表示と停止処理
  Sprite_RollDigit.prototype.update = function() {
    if (this._finished || !this._started) return;

    this._frameCount++;
    const loopFrames = this._loopSpeed;
    const scrollRatio = (this._frameCount % loopFrames) / loopFrames;
    const scrollY = Math.floor(scrollRatio * 10) * this._digitH;
    this.setFrame(0, scrollY % (10 * this._digitH), this._digitW, this._digitH);

    if (this._frameCount >= this._framePerDigit) {
      const finalY = this._targetNumber * this._digitH;
      this.setFrame(0, finalY, this._digitW, this._digitH);
      this._finished = true;
    }
  };
})();
